<?php
// config.php
session_start();

$appName = 'Beatstore Demo PHP v2';

// Archivo donde se guardarán los beats creados desde el admin (JSON)
$beatsFile = __DIR__ . '/data/beats.json';

// --- Definición de beats por defecto (se usan si aún no existe el JSON) ---
$defaultBeats = [
    1 => [
        'id'          => 1,
        'title'       => 'Midnight Vibes',
        'slug'        => 'midnight-vibes',
        'genre'       => 'Lo-fi Hip Hop',
        'bpm'         => 90,
        'base_price'  => 19.00,
        'demo_url'    => 'audio/midnight_vibes_demo.mp3',
        'full_url'    => 'audio/midnight_vibes_full.mp3',
        'description' => 'Beat chill ideal para vlogs nocturnos, podcasts y contenido relajado.',
    ],
    2 => [
        'id'          => 2,
        'title'       => 'Sunny Day',
        'slug'        => 'sunny-day',
        'genre'       => 'Trap / Pop',
        'bpm'         => 140,
        'base_price'  => 29.00,
        'demo_url'    => 'audio/sunny_day_demo.mp3',
        'full_url'    => 'audio/sunny_day_full.mp3',
        'description' => 'Beat energético para campañas, intros y contenido motivacional.',
    ],
];

// --- Definición de licencias ---
$licenseTypes = [
    'basic' => [
        'name'  => 'Licencia Básica',
        'short' => 'Uso en redes sociales y contenido orgánico por 1 año.',
        'multiplier' => 1.0,
    ],
    'premium' => [
        'name'  => 'Licencia Premium',
        'short' => 'Uso comercial en plataformas digitales, monetización y shows en vivo por 1 año.',
        'multiplier' => 2.5,
    ],
    'exclusive' => [
        'name'  => 'Licencia Exclusiva',
        'short' => 'Uso exclusivo, se retira del catálogo. Uso amplio comercial por 1 año.',
        'multiplier' => 6.0,
    ],
];

// --- Helpers generales ---
function e($str) {
    return htmlspecialchars($str, ENT_QUOTES, 'UTF-8');
}

function slugify($text) {
    $text = strtolower($text);
    // Reemplazar acentos básicos
    $replacements = [
        'á' => 'a','é' => 'e','í' => 'i','ó' => 'o','ú' => 'u',
        'ñ' => 'n',
    ];
    $text = strtr($text, $replacements);
    $text = preg_replace('~[^a-z0-9]+~', '-', $text);
    $text = trim($text, '-');
    if ($text === '') {
        $text = 'beat';
    }
    return $text;
}

// --- Manejo de beats en JSON ---

function loadBeats() {
    global $beatsFile, $defaultBeats;

    if (file_exists($beatsFile)) {
        $json = file_get_contents($beatsFile);
        $arr  = json_decode($json, true);
        if (is_array($arr) && count($arr) > 0) {
            $byId = [];
            foreach ($arr as $beat) {
                if (isset($beat['id'])) {
                    $byId[$beat['id']] = $beat;
                }
            }
            return $byId;
        }
    }
    // Si no hay JSON aún, usar los beats por defecto
    return $defaultBeats;
}

function saveBeats($beats) {
    global $beatsFile;
    $dir = dirname($beatsFile);
    if (!is_dir($dir)) {
        @mkdir($dir, 0775, true);
    }
    // Guardar como lista
    $list = array_values($beats);
    file_put_contents($beatsFile, json_encode($list, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Cargamos beats al iniciar
$beats = loadBeats();

// --- Helpers específicos ---

function getAllBeats() {
    global $beats;
    return $beats;
}

function getBeatById($id) {
    global $beats;
    return isset($beats[$id]) ? $beats[$id] : null;
}

function getLicenseByKey($key) {
    global $licenseTypes;
    return isset($licenseTypes[$key]) ? $licenseTypes[$key] : null;
}

function getLicensePrice($beat, $licenseKey) {
    global $licenseTypes;
    if (!isset($licenseTypes[$licenseKey])) return null;
    $mult = $licenseTypes[$licenseKey]['multiplier'];
    return round($beat['base_price'] * $mult, 2);
}

// Devuelve un array con la orden si existe
function findOrderByToken($token) {
    if (!isset($_SESSION['orders']) || !is_array($_SESSION['orders'])) return null;
    foreach ($_SESSION['orders'] as $orderId => $order) {
        if (isset($order['download_token']) && $order['download_token'] === $token) {
            $order['id'] = $orderId;
            return $order;
        }
    }
    return null;
}
